'use client';

import Header from '../../../components/Header';
import MobileHeader from '../../../components/MobileHeader';
import Footer from '../../../components/Footer';
import { useEffect, useRef, useState } from 'react';
import { useTranslations, useLocale } from 'next-intl';
import Link from 'next/link';
import AnimatedText from '../../../components/AnimatedText';
import AnimatedImage from '../../../components/AnimatedImage';
import AnimatedElement from '../../../components/AnimatedElement';

export default function AboutPage() {
  const videoRef = useRef<HTMLVideoElement>(null);
  const [isDesktop, setIsDesktop] = useState(false);
  const t = useTranslations('AboutPage');
  const tCommon = useTranslations('Common');
  const locale = useLocale();

  useEffect(() => {
    const checkScreenSize = () => {
      setIsDesktop(window.innerWidth >= 1024);
    };

    checkScreenSize();
    window.addEventListener('resize', checkScreenSize);

    return () => window.removeEventListener('resize', checkScreenSize);
  }, []);

  useEffect(() => {
    const video = videoRef.current;
    if (video) {
      console.log('About page video element found, attempting to play...');
      video.play().then(() => {
        console.log('About page video is playing successfully');
      }).catch((error) => {
        console.error('Error playing about page video:', error);
      });
    }
  }, []);

  const getLocalizedPath = (path: string) => {
    if (path === '/') {
        return `/${locale}`;
    }
    return `/${locale}${path}`;
  }

  if (isDesktop) {
    // Десктопная версия
    return (
      <div className="flex w-full flex-col items-start bg-[#F5F1F0]">
        <Header />

        {/* Panorama Section with Video Background */}
        <section className="flex flex-col items-center justify-center self-stretch gap-6">
          <div className="relative flex h-[597px] flex-col items-start justify-end self-stretch gap-6 overflow-hidden">
            {/* Background Video */}
            <video
              ref={videoRef}
              autoPlay
              loop
              muted
              playsInline
              controls={false}
              preload="auto"
              className="absolute bottom-0 left-0 h-auto w-full"
              style={{
                position: 'absolute',
                left: 0,
                bottom: 0,
                width: '100%',
                height: 'auto',
                zIndex: 1,
              }}
              onLoadedData={() => console.log('About page video loaded successfully!')}
              onError={(e) => console.error('About page video error:', e)}
              onCanPlay={() => console.log('About page video can play')}
              onPlay={() => console.log('About page video is playing')}
              onLoadedMetadata={() => {
                console.log('About page video metadata loaded');
                const video = videoRef.current;
                if (video) {
                  console.log('Video dimensions:', video.videoWidth, 'x', video.videoHeight);
                  console.log('Container dimensions:', video.clientWidth, 'x', video.clientHeight);
                }
              }}
            >
              <source src="/images/about/watermarked-46c17d41-8bbf-480b-8ef4-210252450ece.mp4" type="video/mp4" />
              Your browser does not support the video tag.
            </video>
            
            {/* Gradient Overlay */}
            <div 
              className="absolute inset-0 w-full h-full"
              style={{
                background: 'linear-gradient(180deg, rgba(0, 0, 0, 0.00) 0%, rgba(0, 0, 0, 0.40) 100%)',
                zIndex: 2,
                pointerEvents: 'none'
              }}
            ></div>
            
            {/* Content Container */}
            <div 
              className="relative z-10 mx-auto flex h-[533px] w-full max-w-[1376px] flex-1 flex-col justify-between items-start p-8"
            >
              <div className="flex w-full flex-1 flex-col items-start justify-between">
                <AnimatedText
                  as="h1"
                  animation="slideInUp"
                  delay={200}
                  className="w-full text-center font-roboto text-[40px] font-light leading-[115%] tracking-[-1.2px] text-white"
                >
                  {t('main_title')}
                </AnimatedText>
                <AnimatedText
                  as="p"
                  animation="slideInUp"
                  delay={400}
                  className="w-full font-roboto text-2xl font-light leading-[130%] text-white"
                >
                  {t('main_subtitle')}
                </AnimatedText>
              </div>
            </div>
          </div>
        </section>

        {/* Main Content Section */}
        <div className="flex py-[150px] px-8 justify-center items-center gap-10 self-stretch bg-white">
          <div className="flex w-full max-w-[1376px] flex-col items-start gap-10">
            <div className="flex flex-col items-start gap-3 self-stretch">
              <AnimatedText
                as="h2"
                animation="slideInUp"
                delay={200}
                className="self-stretch text-[#1C1C1C] font-roboto text-[40px] font-light leading-[115%] tracking-[-1.2px] whitespace-pre-line"
              >
                {t('secondary_title')}
              </AnimatedText>
            </div>
            <AnimatedElement animation="fadeIn" delay={400}>
              <Link href={getLocalizedPath('/lake-token')} className="flex py-3 px-6 justify-center items-center gap-2 rounded-[2px] bg-[#A79D94] hover:bg-[#9a8f86] transition-colors">
                <span className="text-[#F5F1F0] font-roboto text-lg font-normal leading-[150%]">
                  {tCommon('lake_token')}
                </span>
              </Link>
            </AnimatedElement>
          </div>
        </div>

        {/* Cards Section - десктопная версия */}
        <div className="flex py-[60px] flex-col justify-center items-center gap-2 self-stretch">
          <div className="flex flex-row w-full max-w-[1378px] items-start gap-10">
            {/* Card 1 */}
            <div className="flex flex-col items-start flex-1">
              <div className="h-[378px] w-full">
                <AnimatedElement animation="slideInLeft" delay={200} className="w-full h-full">
                  <img
                    src="/images/about/017382eac96af5df1c2166093f6edfa9ff78e491.gif"
                    alt={t('video_fallback')}
                    className="w-full h-full object-cover"
                    style={{
                      width: '100%',
                      height: '100%',
                      objectFit: 'cover',
                      objectPosition: 'center center'
                    }}
                  />
                </AnimatedElement>
              </div>
              <div className="flex pt-6 flex-col items-start gap-4 w-full">
                <AnimatedText
                  as="p"
                  animation="slideInUp"
                  delay={400}
                  className="w-full text-[#1C1C1C] font-roboto text-lg font-light leading-[150%]"
                >
                  {t('card1_text')}
                </AnimatedText>
              </div>
            </div>

            {/* Card 2 */}
            <div className="flex flex-col items-start flex-1">
              <div className="h-[378px] w-full">
                <AnimatedElement animation="slideInRight" delay={400} className="w-full h-full">
                  <video 
                    autoPlay
                    loop
                    muted
                    playsInline
                    controls={false}
                    preload="auto"
                    className="w-full h-full object-cover"
                    style={{
                      width: '100%',
                      height: '100%',
                      objectFit: 'cover',
                      objectPosition: 'center center'
                    }}
                  >
                    <source src="/images/about/doc_2025-07-08_00-44-42.mp4" type="video/mp4" />
                    {t('video_fallback')}
                  </video>
                </AnimatedElement>
              </div>
              <div className="flex pt-6 flex-col items-start gap-4 w-full">
                <AnimatedText
                  as="p"
                  animation="slideInUp"
                  delay={600}
                  className="w-full text-[#1C1C1C] font-roboto text-lg font-light leading-[150%]"
                >
                  {t('card2_text')}
                </AnimatedText>
              </div>
            </div>
          </div>
        </div>

        <Footer />
      </div>
    );
  }

  // Мобильная версия
  return (
    <div className="flex w-full flex-col items-start bg-[#F5F1F0]">
      <MobileHeader />

      {/* Panorama Section with Video Background - мобильная */}
      <section className="flex flex-col justify-center items-center gap-6 self-stretch">
        <div className="flex h-[600px] p-4 flex-col justify-end items-start gap-4 self-stretch relative">
          {/* Background Video */}
          <video 
            ref={videoRef}
            autoPlay
            loop
            muted
            playsInline
            controls={false}
            preload="auto"
            className="absolute inset-0 w-full h-full object-cover"
            style={{
              position: 'absolute',
              top: 0,
              left: 0,
              right: 0,
              bottom: 0,
              width: '100%',
              height: '100%',
              objectFit: 'cover',
              objectPosition: 'center center',
              zIndex: 1
            }}
            onLoadedData={() => console.log('About page video loaded successfully!')}
            onError={(e) => console.error('About page video error:', e)}
            onCanPlay={() => console.log('About page video can play')}
            onPlay={() => console.log('About page video is playing')}
            onLoadedMetadata={() => {
              console.log('About page video metadata loaded');
              const video = videoRef.current;
              if (video) {
                console.log('Video dimensions:', video.videoWidth, 'x', video.videoHeight);
                console.log('Container dimensions:', video.clientWidth, 'x', video.clientHeight);
              }
            }}
          >
            <source src="/images/about/watermarked-46c17d41-8bbf-480b-8ef4-210252450ece.mp4" type="video/mp4" />
            Your browser does not support the video tag.
          </video>
          
          {/* Gradient Overlay */}
          <div 
            className="absolute inset-0 w-full h-full"
            style={{
              background: 'linear-gradient(180deg, rgba(0, 0, 0, 0.00) 0%, rgba(0, 0, 0, 0.40) 100%)',
              zIndex: 2,
              pointerEvents: 'none'
            }}
          ></div>
          
          {/* Content Container */}
          <div 
            className="flex flex-col justify-between items-start flex-1 w-full max-w-[1376px] mx-auto relative h-[320px]"
            style={{ zIndex: 3 }}
          >
            <div className="flex flex-col justify-between items-start flex-1 w-full">
              <h1 className="w-full text-white text-center font-roboto text-2xl font-light leading-[115%] tracking-[-1.2px]">
                {t('main_title')}
              </h1>
              <p className="w-full text-white font-roboto text-base font-light leading-[130%]">
                {t('main_subtitle')}
              </p>
            </div>
          </div>
        </div>
      </section>

      {/* Main Content Section - мобильная */}
      <div className="flex py-8 px-3 justify-center items-center gap-10 self-stretch bg-white">
        <div className="flex w-full max-w-[1376px] flex-col items-start gap-6">
          <div className="flex flex-col items-start gap-3 self-stretch">
            <h2 className="self-stretch text-[#1C1C1C] font-roboto text-2xl font-light leading-[115%] tracking-[-1.2px] whitespace-pre-line">
              {t('secondary_title')}
            </h2>
          </div>
          <Link href={getLocalizedPath('/lake-token')} className="flex w-full py-3 px-6 justify-center items-center gap-2 self-stretch rounded-[2px] bg-[#A79D94] hover:bg-[#9a8f86] transition-colors">
            <span className="text-[#F5F1F0] font-roboto text-lg font-normal leading-[150%]">
              {tCommon('lake_token')}
            </span>
          </Link>
        </div>
      </div>

      {/* Cards Section - мобильная версия */}
      <div className="flex py-8 flex-col justify-center items-center gap-2 self-stretch">
        <div className="flex flex-col w-full max-w-[1378px] px-3 items-start gap-6">
          {/* Card 1 */}
          <div className="flex flex-col items-start w-full">
            <div className="h-[250px] w-full">
              <AnimatedElement animation="slideInUp" delay={200} className="w-full h-full">
                <img
                  src="/images/about/017382eac96af5df1c2166093f6edfa9ff78e491.gif"
                  alt={t('video_fallback')}
                  className="w-full h-full object-cover"
                  style={{
                    width: '100%',
                    height: '100%',
                    objectFit: 'cover',
                    objectPosition: 'center center'
                  }}
                />
              </AnimatedElement>
            </div>
            <div className="flex pt-4 flex-col items-start gap-4 w-full">
              <AnimatedText
                as="p"
                animation="slideInUp"
                delay={400}
                className="w-full text-[#1C1C1C] font-roboto text-base font-light leading-[150%]"
              >
                {t('card1_text')}
              </AnimatedText>
            </div>
          </div>

          {/* Card 2 */}
          <div className="flex flex-col items-start w-full">
            <div className="h-[250px] w-full">
              <AnimatedElement animation="slideInUp" delay={400} className="w-full h-full">
                <video 
                  autoPlay
                  loop
                  muted
                  playsInline
                  controls={false}
                  preload="auto"
                  className="w-full h-full object-cover"
                  style={{
                    width: '100%',
                    height: '100%',
                    objectFit: 'cover',
                    objectPosition: 'center center'
                  }}
                >
                  <source src="/images/about/doc_2025-07-08_00-44-42.mp4" type="video/mp4" />
                  {t('video_fallback')}
                </video>
              </AnimatedElement>
            </div>
            <div className="flex pt-4 flex-col items-start gap-4 w-full">
              <AnimatedText
                as="p"
                animation="slideInUp"
                delay={600}
                className="w-full text-[#1C1C1C] font-roboto text-base font-light leading-[150%]"
              >
                {t('card2_text')}
              </AnimatedText>
            </div>
          </div>
        </div>
      </div>

      <Footer />
    </div>
  );
} 